(function() {
    "use strict";

    // Utility functions
    const isUndefined = function(t) { return t === undefined; };
    const isFunction = function(t) { return typeof t === "function"; };
    const isString = function(t) { return typeof t === "string"; };
    const clone = function(t) { return JSON.parse(JSON.stringify(t)); };
    const isObjectNotEmpty = function(t) {
        if (!t || !t.constructor || t.nodeType) return false;
        try { return "{}" !== JSON.stringify(t); } catch (e) { return true; }
    };

    const merge = function() {
        let args = Array.from(arguments), dest = args[0] || {}, i = 1, len = args.length, deep = false;
        if (typeof dest === "boolean") { deep = dest; dest = args[i] || {}; i++; }
        if (typeof dest !== "object" && !isFunction(dest)) dest = {};
        if (i === len) { dest = this; i--; }
        for (; i < len; i++) {
            if (!isUndefined(args[i]) && args[i] !== null) {
                for (let key of Object.keys(args[i])) {
                    if (Object.prototype.hasOwnProperty.call(args[i], key)) {
                        let src = args[i][key];
                        if (key === "__proto__" || dest === src) continue;
                        let isArraySrc = Array.isArray(src);
                        if (deep && src && (isObjectNotEmpty(src) || isArraySrc)) {
                            let cloneSrc = dest[key];
                            let newSrc = isArraySrc && !Array.isArray(cloneSrc) ? [] : (isArraySrc || isObjectNotEmpty(cloneSrc) ? cloneSrc : {});
                            dest[key] = merge(deep, newSrc, src);
                        } else if (!isUndefined(src)) {
                            dest[key] = src;
                        }
                    }
                }
            }
        }
        return dest;
    };

    const mod = function(t, n) { return Math.abs(t - n * Math.floor(t / n)); };
    const pad = function(t, n = 2) {
        let str = String(Math.abs(t)), len = str.length, result = "";
        if (t < 0) result += "-";
        while (len < n) { len++; result += "0"; }
        return result + str;
    };

    const validateDate = function(t, input, defaults) {
        let merged = merge(input, defaults);
        let init = t.initDate, max = t.options.maxDate, min = t.options.minDate;
        let year = merged.year, month = merged.month, day = merged.day;
        year = isNaN(year) || year < 1000 || year > 9999 ? init.year : year < min.year ? (month = 1, min.year) : year > max.year ? max.year : year;
        month = isNaN(month) || month < 1 || month > 12 ? init.month : year <= min.year && month < min.month ? (day = 1, min.month) : year >= max.year && month > max.month ? max.month : month;
        day = isNaN(day) || day < 1 ? init.day : month <= min.month && day < min.day ? min.day : month >= max.month && day > max.day ? max.day : day;
        return { year: parseInt(year), month: parseInt(month), day: parseInt(day) };
    };

    const validateTime = function(t, input, defaults) {
        let merged = merge(input, defaults);
        let init = t.initTime, max = t.options.maxTime, min = t.options.minTime;
        let hour = merged.hour, minute = merged.minute, second = merged.second;
        hour = isNaN(hour) || hour < 0 || hour > 23 ? init.hour : hour < min.hour ? min.hour : hour > max.hour ? max.hour : hour;
        minute = isNaN(minute) || minute < 0 || minute > 59 ? init.minute : hour <= min.hour && minute < min.minute ? min.minute : hour >= max.hour && minute > max.minute ? max.minute : minute;
        second = isNaN(second) || second < 0 || second > 59 ? init.second : hour <= min.hour && minute <= min.minute && second < min.second ? min.second : hour >= max.hour && minute >= max.minute && second > max.second ? max.second : second;
        return { hour: parseInt(hour), minute: parseInt(minute), second: parseInt(second) };
    };

    const isDateValid = function(t, year, month, day) {
        let min = t.options.minDate, max = t.options.maxDate;
        let dateStr = formatDate(t, { year, month, day });
        min = isObjectNotEmpty(min) ? formatDate(t, { year: min.year, month: min.month, day: min.day }) : dateStr;
        max = isObjectNotEmpty(max) ? formatDate(t, { year: max.year, month: max.month, day: max.day }) : dateStr;
        return dateStr <= max && dateStr >= min;
    };

    const parseInput = function(t, input) {
        let sep = t.options.separatorChars;
        let parts = input.split(sep.between);
        let date = t.options.date ? parts[0].split(sep.date) : {};
        let time = t.options.date ? (t.options.time && parts[1] ? parts[1].split(sep.time) : {}) : parts[0].split(sep.time);
        return {
            year: parseInt(date[0]),
            month: parseInt(date[1]),
            day: parseInt(date[2]),
            hour: parseInt(time[0]),
            minute: parseInt(time[1]),
            second: parseInt(time[2])
        };
    };

    const formatDate = function(t, date) {
        let sep = t.options.separatorChars;
        return `${date.year}${sep.date}${pad(date.month)}${sep.date}${pad(date.day)}`;
    };

    const isValidDateFormat = function(t, input) {
        if (!input) return false;
        let parts = input.substr(0, 10).split(t.options.separatorChars.date);
        return parts.length === 3 && parts[0].length === 4 && parts[1].length === 2 && parts[2].length === 2;
    };

    const isValidTimeFormat = function(t, input) {
        if (!input) return false;
        let start = t.options.date ? 11 : 0;
        let parts = input.substr(start, 8).split(t.options.separatorChars.time);
        return parts.length === (t.options.hasSecond ? 3 : 2) && !parts.find(t => t.toString().length !== 2);
    };

    // Selector constants
    const PREFIX = "gdp";
    const CONTAINER_CLASS = `${PREFIX}-container`;
    const OVERLAY_CLASS = `${PREFIX}-overlay`;
    const YEARS_CLASS = `div.${PREFIX}-years`;
    const YEAR_CLASS = `div.${PREFIX}-year`;
    const MONTHS_CLASS = `div.${PREFIX}-months`;
    const MONTH_CLASS = `div.${PREFIX}-month`;
    const DAYS_CLASS = `div.${PREFIX}-days`;
    const DAY_CLASS = `div.${PREFIX}-day`;
    const NOT_IN_MONTH_CLASS = `div.${PREFIX}-day.not-in-month`;
    const DISABLED_DAY_CLASS = `div.${PREFIX}-day.disabled-day`;
    const DISABLED_NOT_IN_MONTH_CLASS = `${NOT_IN_MONTH_CLASS}.disabled-day`;
    const DAY_NAME_CLASS = `div.${PREFIX}-day-name`;
    const PLUS_CLASS = `div.${PREFIX}-icon-plus`;
    const MINUS_CLASS = `div.${PREFIX}-icon-minus`;
    const FOOTER_CLASS = `div.${PREFIX}-footer`;
    const TODAY_BTN_CLASS = `div.${PREFIX}-btn-today`;
    const EMPTY_BTN_CLASS = `div.${PREFIX}-btn-empty`;
    const CLOSE_BTN_CLASS = `div.${PREFIX}-btn-close`;
    const TIME_CONTAINER_CLASS = `div.${PREFIX}-time-container`;
    const TIME_CLASS = `div.${PREFIX}-time`;
    const NOT_IN_RANGE_CLASS = "not-in-range";
    const HOLIDAY_CLASS = "holiday";
    const CHANGE_EVENT = `${PREFIX}:change`;
    const CLICK_EVENT = "click";
    const FOCUSIN_EVENT = "focusin";
    const KEYDOWN_EVENT = "keydown";
    const TODAY = "today";
    const ATTR = "attr";
    const ONLY_DATE_ATTR = "data-gdp-only-date";
    const ONLY_TIME_ATTR = "data-gdp-only-time";
    const VISIBLE = "visible";
    const DISPLAY_BLOCK = "block";
    const DISPLAY_NONE = "none";

    const getScrollParent = function(node) {
        if (["html", "body", "#document"].indexOf((node.nodeName || "").toLowerCase()) >= 0) return window;
        if (node instanceof HTMLElement) {
            let style = window.getComputedStyle(node);
            if (/auto|scroll|overlay/.test(style.overflow + style.overflowX + style.overflowY)) return node;
        }
        return getScrollParent(node.parentNode);
    };

    const createEvent = function(name) {
        let event = document.createEvent("Event");
        event.initEvent(name, true, true);
        return event;
    };

    const dispatchEvent = function(element, eventName) {
        if (element) {
            element.dispatchEvent(createEvent(eventName));
            if (eventName === CHANGE_EVENT) {
                element.dispatchEvent(createEvent("change"));
                element.dispatchEvent(createEvent("input"));
            }
        }
    };

    const createElement = function(tagAndClasses, parent, event, handler, content) {
        const parts = tagAndClasses.split(".");
        const tag = parts.shift() || "div";
        const classes = parts;
        const element = window.document.createElement(tag);

        if (isString(parent)) {
            window.document.querySelector(parent).appendChild(element);
        } else {
            parent.appendChild(element);
        }

        if (classes.length) {
            element.className = classes.join(" ");
        }

        if (event && handler) {
            const events = event.split(" ");
            for (let i = 0; i < events.length; i++) {
                element.addEventListener(events[i], handler, false);
            }
        }

        if (!isUndefined(content)) {
            element.innerHTML = content;
        }

        return element;
    };

    const setInnerHTML = function(element, content) {
        element.innerHTML = content;
    };

    const isLeapYear = function(year) {
        return (year % 4 === 0 && year % 100 !== 0) || year % 400 === 0;
    };

    const getDayOfWeek = function(year, month, day) {
        return new Date(year, month - 1, day).getDay();
    };

    const getDaysInMonth = function(year, month) {
        return [0, 31, 28 + isLeapYear(year), 31, 30, 31, 30, 31, 31, 30, 31, 30, 31][month];
    };

    const range = function(start, end) {
        let result = [];
        for (let i = start; i <= end; i++) result.push(pad(i));
        return result;
    };

    const createTimeSelect = function(t, parent, type) {
        let selectContainer = createElement(TIME_CLASS, parent);
        let select = createElement("select", selectContainer, "change", function(e) {
            let value = {};
            value[type] = e.target.value;
            t.setValue(validateTime(t, t.initTime, value));
        });
        select.tabIndex = -1;

        let min = merge({ hour: 0, minute: 0, second: 0 }, t.options.minTime);
        let max = merge({ hour: 23, minute: 59, second: 59 }, t.options.maxTime);
        let options;
        if (type === "hour") {
            options = range(min.hour, max.hour);
        } else if (type === "minute") {
            options = min.hour === max.hour ? range(min.minute, max.minute) :
                     t.initTime.hour === min.hour ? range(min.minute, 59) :
                     t.initTime.hour === max.hour ? range(0, max.minute) : range(0, 59);
        } else {
            options = min.hour === max.hour && min.minute === max.minute ? range(min.second, max.second) :
                     t.initTime.hour === min.hour && t.initTime.minute === min.minute ? range(min.second, 59) :
                     t.initTime.hour === max.hour && t.initTime.minute === max.minute ? range(0, max.second) : range(0, 59);
        }

        for (let i = 0; i < options.length; i++) {
            let option = createElement("option", select);
            option.value = options[i];
            option.text = options[i];
            option.selected = parseInt(options[i]) === parseInt(t.getValue[type] || t.initTime[type]);
        }
    };

    const getDayClass = function(day) {
        return day === 6 ? `.last-week.${HOLIDAY_CLASS}` : "";
    };

    const createPlusIcon = function(t, parent, isYear) {
        let className = "";
        if (isYear) {
            if (isObjectNotEmpty(t.options.maxDate) && t.initDate.year >= t.options.maxDate.year) {
                className = `.${NOT_IN_RANGE_CLASS}`;
            }
        } else {
            if (isObjectNotEmpty(t.options.maxDate) && 
                (t.initDate.year > t.options.maxDate.year || 
                 (t.initDate.year === t.options.maxDate.year && t.initDate.month >= t.options.maxDate.month))) {
                className = `.${NOT_IN_RANGE_CLASS}`;
            }
        }
        createElement(`${PLUS_CLASS}${className}`, parent, CLICK_EVENT, isYear ? () => {
            console.log("Plus year clicked, current year:", t.initDate.year);
            t.increaseYear();
        } : () => {
            console.log("Plus month clicked, current month:", t.initDate.month);
            t.increaseMonth();
        }, t.options.plusHtml);
    };

    const createMinusIcon = function(t, parent, isYear) {
        let className = "";
        if (isYear) {
            if (isObjectNotEmpty(t.options.minDate) && t.initDate.year <= t.options.minDate.year) {
                className = `.${NOT_IN_RANGE_CLASS}`;
            }
        } else {
            if (isObjectNotEmpty(t.options.minDate) && 
                (t.initDate.year < t.options.minDate.year || 
                 (t.initDate.year === t.options.minDate.year && t.initDate.month <= t.options.minDate.month))) {
                className = `.${NOT_IN_RANGE_CLASS}`;
            }
        }
        createElement(`${MINUS_CLASS}${className}`, parent, CLICK_EVENT, isYear ? () => {
            console.log("Minus year clicked, current year:", t.initDate.year);
            t.decreaseYear();
        } : () => {
            console.log("Minus month clicked, current month:", t.initDate.month);
            t.decreaseMonth();
        }, t.options.minusHtml);
    };

    const createYearPicker = function(t) {
        let container = createElement(YEARS_CLASS, t.dpContainer);
        createPlusIcon(t, container, true);
        let yearContainer = createElement(YEAR_CLASS, container);
        createMinusIcon(t, container, true);
        let useDropDown = t.options.useDropDownYears;
        let input = createElement(useDropDown ? "select" : "input", yearContainer, "keyup change", function(e) {
            if (e.target.value < 1000 || e.target.value > 9999) return;
            t.yearChange(e.target.value);
        });

        if (useDropDown) {
            let { min, max } = (function(t) {
                let year = t.initDate.year;
                return { min: isObjectNotEmpty(t.options.minDate) ? t.options.minDate.year : Math.round(year - 200, -2), 
                         max: isObjectNotEmpty(t.options.maxDate) ? t.options.maxDate.year : Math.round(year + 200, 2) };
            })(t);
            for (let i = min; i <= max; i++) {
                let option = createElement("option", input);
                option.value = i;
                option.text = i;
                option.selected = i === t.initDate.year;
            }
        } else {
            input.tabIndex = -1;
            input.value = t.initDate.year;
            input.type = "number";
        }
    };

    const createMonthPicker = function(t) {
        let container = createElement(MONTHS_CLASS, t.dpContainer);
        createPlusIcon(t, container, false);
        let monthContainer = createElement(MONTH_CLASS, container);
        createMinusIcon(t, container, false);
        let select = createElement("select", monthContainer, "change", function(e) {
            t.monthChange(e.target.value);
        });
        select.tabIndex = -1;

        let months = (function(t) {
            let year = t.initDate.year, min = t.options.minDate, max = t.options.maxDate;
            let start = 1, end = 12;
            if (isObjectNotEmpty(min) && year === min.year) {
                start = min.month;
                if (isObjectNotEmpty(max) && year === max.year) end = max.month;
            } else if (isObjectNotEmpty(max) && year === max.year) {
                start = 1;
                end = max.month;
            }
            let result = [];
            for (let i = start; i <= end; i++) result.push(i);
            return result;
        })(t);

        let monthNames = t.options.months;
        for (let i = 0; i < months.length; i++) {
            let option = createElement("option", select);
            option.value = months[i];
            option.text = monthNames[months[i] - 1];
            option.selected = months[i] === t.initDate.month;
        }
    };

    const createDayPicker = function(t) {
        let container = createElement(DAYS_CLASS, t.dpContainer);
        for (let i = 0; i < 7; i++) {
            createElement(`${DAY_NAME_CLASS}${getDayClass(i)}`, container, null, null, t.options.days[i]);
        }

        let state = { day: 1, inBeforeMonth: false, inAfterMonth: false, isValid: false, isHoliday: false, className: "", year: t.initDate.year, month: t.initDate.month };
        const resetState = function(s) {
            if (!s.day || s.inBeforeMonth) s.day = 1;
            else s.day += 1;
            s.inBeforeMonth = false;
            s.inAfterMonth = false;
            s.isValid = false;
            s.isHoliday = false;
            s.className = "";
            s.year = t.initDate.year;
            s.month = t.initDate.month;
            return s;
        };

        let daysInMonth = getDaysInMonth(state.year, state.month);
        let firstDayOfWeek = getDayOfWeek(state.year, state.month, 1);
        let totalDays = 7 * Math.ceil((firstDayOfWeek + daysInMonth) / 7) - 1;
        let prevMonth = state.month === 1 ? 12 : state.month - 1;
        let nextMonth = state.month === 12 ? 1 : state.month + 1;
        let prevYear = prevMonth === 12 ? state.year - 1 : state.year;
        let nextYear = nextMonth === 1 ? state.year + 1 : state.year;
        let daysInPrevMonth = prevMonth === 1 ? getDaysInMonth(state.year - 1, prevMonth) : getDaysInMonth(state.year, prevMonth);
        let offset = daysInPrevMonth - firstDayOfWeek;
        let afterOffset = 0;

        for (let i = 0; i <= totalDays; i++) {
            state.inBeforeMonth = state.day <= firstDayOfWeek && i < firstDayOfWeek;
            state.inAfterMonth = i >= daysInMonth + firstDayOfWeek;
            if (state.inBeforeMonth || state.inAfterMonth) {
                if (state.inBeforeMonth) {
                    offset++;
                    state.day = offset;
                    state.year = prevYear;
                    state.month = prevMonth;
                } else {
                    afterOffset++;
                    state.day = afterOffset;
                    state.year = nextYear;
                    state.month = nextMonth;
                }
            }
            state.isValid = isDateValid(t, state.year, state.month, state.day);
            state.className = getDayClass(getDayOfWeek(state.year, state.month, state.day));
            if (t.inputValue.day === state.day && t.inputValue.year === state.year && t.inputValue.month === state.month) {
                state.className += ".selected";
            }
            if (t.today.day === state.day && t.today.year === state.year && t.today.month === state.month) {
                state.className += ".today";
            }
            if (isFunction(t.options.dayRendering)) {
                state = merge(state, t.options.dayRendering(state, t.input));
            }
            if (state.isHoliday) {
                state.className += `.${HOLIDAY_CLASS}`;
            }

            let className = state.isValid ? DAY_CLASS : DISABLED_DAY_CLASS;
            if (state.inBeforeMonth || state.inAfterMonth) {
                className = NOT_IN_MONTH_CLASS;
                if (!state.isValid) className = DISABLED_NOT_IN_MONTH_CLASS;
            }

            let dayElement = createElement(`${className}${state.className}`, container, null, null, state.day);
            dayElement.day = state.day;
            dayElement.month = state.month;
            dayElement.year = state.year;
            if (state.isValid) {
                dayElement.addEventListener(CLICK_EVENT, () => t.setValue({ year: dayElement.year, month: dayElement.month, day: dayElement.day }));
            }
            state = resetState(state);
        }
    };

    const createFooter = function(t) {
        let footer = createElement(FOOTER_CLASS, t.dpContainer);
        if (t.options.showTodayBtn && t.options.date && isDateValid(t, t.today.year, t.today.month, t.today.day)) {
            createElement(TODAY_BTN_CLASS, footer, CLICK_EVENT, () => t.setValue(t.today), "Today");
        } else if (!t.options.date || t.options.time) {
            createElement(TODAY_BTN_CLASS, footer, CLICK_EVENT, () => { t.setValue(t.initTime); t.hide(); }, "Select");
        }
        if (t.options.showEmptyBtn) {
            createElement(EMPTY_BTN_CLASS, footer, CLICK_EVENT, () => {
                t.input.value = "";
                dispatchEvent(t.input, CHANGE_EVENT);
                if (t.options.hideAfterChange) t.hide();
            }, "Clear");
        }
        if (t.options.showCloseBtn) {
            createElement(CLOSE_BTN_CLASS, footer, CLICK_EVENT, () => t.hide(), "Close");
        }
    };

    const drawPicker = function(t) {
        setInnerHTML(t.dpContainer, "");
        if (t.options.date) {
            createYearPicker(t);
            createMonthPicker(t);
            createDayPicker(t);
        }
        if (t.options.time) {
            let timeClass = TIME_CONTAINER_CLASS + (t.options.time && !t.options.date ? ".gdp-only-time" : "");
            let timeContainer = createElement(timeClass, t.dpContainer);
            if (t.options.hasSecond) createTimeSelect(t, timeContainer, "second");
            createTimeSelect(t, timeContainer, "minute");
            createTimeSelect(t, timeContainer, "hour");
        }
        createFooter(t);
    };

    const isMobile = /iphone|ipod|android|ie|blackberry|fennec/.test(window.navigator?.userAgent?.toLowerCase());

    const gregorianDatepicker = {
        options: clone({
            days: ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"],
            months: ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"],
            initDate: null,
            today: null,
            initTime: null,
            hasSecond: true,
            time: false,
            date: true,
            minDate: { year: 1900, month: 1, day: 1 },
            maxDate: { year: 2100, month: 12, day: 31 },
            minTime: { hour: 0, minute: 0, second: 0 },
            maxTime: { hour: 23, minute: 59, second: 59 },
            separatorChars: { date: "/", between: " ", time: ":" },
            zIndex: 1000,
            container: "body",
            selector: "input[data-gdp]",
            autoShow: true,
            autoHide: true,
            hideAfterChange: true,
            plusHtml: '<svg viewBox="0 0 1024 1024"><g><path d="M810 554h-256v256h-84v-256h-256v-84h256v-256h84v256h256v84z"></path></g></svg>',
            minusHtml: '<svg viewBox="0 0 1024 1024"><g><path d="M810 554h-596v-84h596v84z"></path></g></svg>',
            changeMonthRotateYear: false,
            showTodayBtn: true,
            showEmptyBtn: true,
            showCloseBtn: isMobile,
            autoReadOnlyInput: isMobile,
            useDropDownYears: true,
            topSpace: 0,
            bottomSpace: 0,
            overflowSpace: -10
        }),

        input: null,
        isTransitioning: false,

        get dpContainer() {
            if (!this._dpContainer || !this._dpContainer.isConnected) {
                this._dpContainer = createElement(CONTAINER_CLASS, this.options.container);
                this._dpContainer.style.zIndex = this.options.zIndex;
            }
            if (!this.overlayElm || !this.overlayElm.isConnected) {
                this.overlayElm = createElement(OVERLAY_CLASS, this.options.container);
                this.overlayElm.style.zIndex = this.options.zIndex - 1;
            }
            return this._dpContainer;
        },

        get today() {
            if (!this._today) {
                this._today = this.options.today || (function() {
                    let date = new Date();
                    return { year: date.getFullYear(), month: date.getMonth() + 1, day: date.getDate() };
                })();
            }
            return this._today;
        },

        get inputValue() {
            let value = clone(this.input.value);
            return (function(t, input) {
                if (!input) return false;
                let sep = t.options.separatorChars;
                let date = t.options.date ? `\\d{4}${sep.date}\\d{2}${sep.date}\\d{2}` : "";
                let time = t.options.time ? `\\d{2}${sep.time}\\d{2}${t.options.hasSecond ? `${sep.time}\\d{2}` : ""}` : "";
                return new RegExp(date + (date && time ? sep.between : "") + time).test(input);
            })(this, value) || (isString(value) && isValidDateFormat(this, value)) ? parseInput(this, value) : {};
        },

        get initDate() {
            if (this.options.initDate) return this.options.initDate;
            if (!this._initDate) {
                this._initDate = clone(this.input.value) || {};
                if (isObjectNotEmpty(this._initDate)) {
                    this._initDate = this.options.initDate || clone(this.today);
                } else if (isString(this._initDate) && isValidDateFormat(this, this._initDate)) {
                    this._initDate = parseInput(this, this._initDate);
                } else {
                    this._initDate = clone(this.today);
                }
                this._initDate = validateDate(this, this._initDate);
            }
            return this._initDate;
        },

        get initTime() {
            if (this._initTime) return this._initTime;
            let date = new Date();
            let time = { hour: date.getHours(), minute: date.getMinutes(), second: 0 };
            this._initTime = clone(this.input.value) || this.options.initTime || time;
            if (isString(this._initTime) && isValidTimeFormat(this, this._initTime)) {
                this._initTime = parseInput(this, this._initTime);
            } else {
                this._initTime = time;
            }
            this._initTime = validateTime(this, this._initTime);
            return this._initTime;
        },

        _draw: function() {
            drawPicker(this);
        },

        show: function(input) {
            this._initDate = null;
            this._initTime = null;
            this._value = null;
            this.input = input;
            this._draw();
            if (this.options.autoReadOnlyInput && !input.readOnly) {
                input.setAttribute("readonly", "readonly");
                input.readOnly = true;
            }
            this.isTransitioning = true;
            this.dpContainer.style.visibility = VISIBLE;
            this.dpContainer.style.display = DISPLAY_BLOCK;
            this.overlayElm.style.display = DISPLAY_BLOCK;
            setTimeout(() => {
                this.dpContainer.style.visibility = VISIBLE;
                this.dpContainer.style.display = DISPLAY_BLOCK;
                this.overlayElm.style.display = DISPLAY_BLOCK;
                this.isShow = true;
                this.isTransitioning = false;
            }, 300);
            this.setPosition();
            getScrollParent(input).addEventListener("scroll", () => this.setPosition(), { passive: true });
            document.addEventListener(KEYDOWN_EVENT, handleKeydown);
        },

        hide: function() {
            this.dpContainer.style.visibility = "hidden";
            this.dpContainer.style.display = DISPLAY_NONE;
            this.overlayElm.style.display = DISPLAY_NONE;
            this.isShow = false;
            document.removeEventListener(KEYDOWN_EVENT, handleKeydown);
        },

        setPosition: function() {
            if (this.dpContainer.style.visibility === VISIBLE) {
                let rect = this.input.getBoundingClientRect();
                let inputHeight = rect.height, left = rect.left, top = rect.top + inputHeight;
                top += this.options.topSpace;
                let bodyWidth = window.document.body.offsetWidth;
                let containerWidth = this.dpContainer.offsetWidth;
                let containerHeight = this.dpContainer.offsetHeight;
                if (left + containerWidth >= bodyWidth) {
                    left -= left + containerWidth - (bodyWidth + this.options.overflowSpace);
                }
                if (top - inputHeight >= containerHeight && top + containerHeight >= window.innerHeight) {
                    top -= containerHeight + inputHeight + this.options.bottomSpace + this.options.topSpace;
                }
                this.dpContainer.style.position = "fixed";
                this.dpContainer.style.left = left + "px";
                this.dpContainer.style.top = top + "px";
            }
        },

        get getValue() {
            this._value = this._value || this.inputValue || {};
            return this._value;
        },

        setValue: function(value) {
            this._value = merge({ year: this.today.year, month: this.today.month, day: this.today.day, hour: this.initTime.hour, minute: this.initTime.minute, second: this.initTime.second }, this._value, value);
            this._initTime = null;
            let sep = this.options.separatorChars;
            let dateStr = this.options.date ? `${this._value.year}${sep.date}${pad(this._value.month)}${sep.date}${pad(this._value.day)}` : "";
            let timeStr = this.options.time ? `${pad(this._value.hour)}${sep.time}${pad(this._value.minute)}${this.options.hasSecond ? sep.time + pad(this._value.second) : ""}` : "";
            this.input.value = dateStr + (dateStr && timeStr ? sep.between : "") + timeStr;
            dispatchEvent(this.input, CHANGE_EVENT);
            if (!this.options.time && this.options.hideAfterChange) {
                this.hide();
            } else {
                this._draw();
            }
        },

        increaseMonth: function() {
            console.log("Increasing month, current:", this._initDate);
            let isLastMonth = this._initDate.month === 12;
            if (this.options.changeMonthRotateYear && isLastMonth) this.increaseYear();
            this.monthChange(isLastMonth ? 1 : this._initDate.month + 1);
        },

        decreaseMonth: function() {
            console.log("Decreasing month, current:", this._initDate);
            let isFirstMonth = this._initDate.month === 1;
            if (this.options.changeMonthRotateYear && isFirstMonth) this.decreaseYear();
            this.monthChange(isFirstMonth ? 12 : this._initDate.month - 1);
        },

        monthChange: function(month) {
            console.log("Changing month to:", month);
            this._initDate = validateDate(this, this._initDate, { month });
            this._draw();
        },

        increaseYear: function() {
            console.log("Increasing year, current:", this._initDate.year);
            this.yearChange(this._initDate.year + 1);
        },

        decreaseYear: function() {
            console.log("Decreasing year, current:", this._initDate.year);
            this.yearChange(this._initDate.year - 1);
        },

        yearChange: function(year) {
            console.log("Changing year to:", year);
            this._initDate = validateDate(this, this._initDate, { year });
            this._draw();
        },

        init: function(options) {
            this.updateOptions(options);
            Element.prototype.matches = Element.prototype.matchesSelector || Element.prototype.mozMatchesSelector || Element.prototype.msMatchesSelector || Element.prototype.oMatchesSelector || Element.prototype.webkitMatchesSelector;
            window.addEventListener("resize", () => this.setPosition());
            if (this.options.autoHide) {
                document.body.addEventListener(CLICK_EVENT, (e) => {
                    if (this.isShow && !this.dpContainer.contains(e.target) && e.target !== this.input) {
                        this.hide();
                    }
                });
            }
            if (this.options.autoShow) {
                document.body.addEventListener(FOCUSIN_EVENT, (e) => {
                    if (e.target && e.target.matches(this.options.selector)) {
                        this.show(e.target);
                    }
                });
            }
        },

        updateOptions: function(options) {
            this.options = updateOptions(options);
        }
    };

    const getAttributeValue = function(attr, isTime) {
        let value = gregorianDatepicker.input?.getAttribute(attr);
        if (!isTime && value === TODAY) return clone(gregorianDatepicker.today);
        if (!isString(value)) return {};
        try { value = document.querySelector(value).value; } catch (e) {}
        return isTime ? (isValidTimeFormat(gregorianDatepicker, value) ? parseInput(gregorianDatepicker, value) : {}) :
                       (isValidDateFormat(gregorianDatepicker, value) ? parseInput(gregorianDatepicker, value) : {});
    };

    const updateOptions = function(options) {
        if (!isUndefined(gregorianDatepicker.options._date) && isUndefined(options.date)) options.date = gregorianDatepicker.options._date;
        if (!isUndefined(gregorianDatepicker.options._time) && isUndefined(options.time)) options.time = gregorianDatepicker.options._time;
        options.separatorChars = merge(gregorianDatepicker.options.separatorChars, options.separatorChars);
        options = merge({}, gregorianDatepicker.options, options);
        if (options.minDate === TODAY) options.minDate = clone(gregorianDatepicker.today);
        if (options.maxDate === TODAY) options.maxDate = clone(gregorianDatepicker.today);
        if (options.initDate === ATTR || options._initDateIsAttr) {
            delete options.initDate;
            options._initDateIsAttr = true;
            Object.defineProperty(options, "initDate", { get: () => getAttributeValue("data-gdp-init-date"), enumerable: true });
        }
        if (options.minDate === ATTR || options._minDateIsAttr) {
            delete options.minDate;
            options._minDateIsAttr = true;
            Object.defineProperty(options, "minDate", { get: () => getAttributeValue("data-gdp-min-date"), enumerable: true });
        }
        if (options.maxDate === ATTR || options._maxDateIsAttr) {
            delete options.maxDate;
            options._maxDateIsAttr = true;
            Object.defineProperty(options, "maxDate", { get: () => getAttributeValue("data-gdp-max-date"), enumerable: true });
        }
        if (options.minTime === ATTR || options._minTimeIsAttr) {
            delete options.minTime;
            options._minTimeIsAttr = true;
            Object.defineProperty(options, "minTime", { get: () => getAttributeValue("data-gdp-min-time", true), enumerable: true });
        }
        if (options.maxTime === ATTR || options._maxTimeIsAttr) {
            delete options.maxTime;
            options._maxTimeIsAttr = true;
            Object.defineProperty(options, "maxTime", { get: () => getAttributeValue("data-gdp-max-time", true), enumerable: true });
        }
        options._date = options.date;
        delete options.date;
        Object.defineProperty(options, "date", {
            get: function() {
                return !(gregorianDatepicker.input?.hasAttribute(ONLY_TIME_ATTR)) && (options._date || gregorianDatepicker.input?.hasAttribute(ONLY_DATE_ATTR));
            },
            enumerable: true
        });
        options._time = options.time;
        delete options.time;
        Object.defineProperty(options, "time", {
            get: function() {
                return !(gregorianDatepicker.input?.hasAttribute(ONLY_DATE_ATTR)) && (options._time || gregorianDatepicker.input?.hasAttribute(ONLY_TIME_ATTR));
            },
            enumerable: true
        });
        return options;
    };

    function handleKeydown(e) {
        if (e.key === "Escape" && !gregorianDatepicker.isTransitioning) {
            gregorianDatepicker.input?.blur();
            gregorianDatepicker.hide();
        }
    }

    window.gregorianDatepicker = {
        startWatch: function(options = {}) {
            gregorianDatepicker.init(options);
        },
        show: function(input) {
            gregorianDatepicker.show(input);
        },
        hide: function() {
            gregorianDatepicker.hide();
        },
        updateOptions: function(options) {
            gregorianDatepicker.updateOptions(options);
        }
    };
})();